/*
fornax useless machine advanced edition

Copyright (C) 2009 Andreas Fiessler aka fornax

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <avr/io.h>
#include <inttypes.h>
#include <avr/interrupt.h>
#include <util/delay.h>

#include "lcdlib.h"



#define OCLEFT OCR1A
#define OCRIGHT OCR1B
#define OCSERVO OCR2

#define PHASE_A	(PINB & 1<<PB6)
#define PHASE_B (PINB & 1<<PB7)
#define DEBOUNCE 30

/* controller parameters */
#define PID_KP 	6 // was 7
#define PID_KI	0 // works without integral
#define PID_KD 	9 // was 12	
/*
 * NOTE: to adjust KI and KD for sampling rate: 
 * KI = KI/rate
 * KD = KD*rate
 */

#define SERVOLIDPIN	PB0

/* not used, just for docs */
#define SW0	PD7
#define SW1	PD0
#define SW2	PC0
#define SW3	PC1
#define SW4	PC2
#define SW5	PC3
#define SW6	PC4
#define SW7	PC5


#define SERVO_IDLE	OCSERVO=34
#define SERVO_TOG	OCSERVO=15
#define	SERVO_HOLD	OCSERVO=20

#define LIDOPEN		lid_pos=75
#define LIDCLOSE	lid_pos=110

/* 2367 steps per 10cm */
/* position of switches */
#define POS0	954
#define POS1 	1683
#define POS2	2336
#define POS3	3083
#define POS4	3807
#define POS5	4479
#define POS6	5173
#define POS7	5901

/* tolerance +/- of position for activating servo */
#define POSTOL	15 

/* delay times for switches, see main */
#define WAIT0	6
#define WAIT1	10


volatile int16_t position;
volatile int16_t position_set;
volatile uint8_t lid_pos;
static int8_t last;

/* encoder readout, thx to P. Dannegger */
void encode_init( void )
{
	int8_t new;
	new = 0;
	if( PHASE_A )
	    new = 3;
	if( PHASE_B )
	    new ^= 1;
	last = new;
	position = 0;
	TCCR0 |=  (1 << CS00);
	TIMSK |= 1<<TOIE0;
}

void init( void )
{
	/* PWM outputs */
	DDRB |= (1 << PB0) | (1 << PB1) | (1 << PB2) | (1 << PB3);
	/* high side */
	DDRB |= (1 << PB4) | (1 << PB5);
	
	/* pullups for switches */
	PORTC |= (1 << PC0) | (1 << PC1) | (1 << PC2) | (1 << PC3) | (1 << PC4) | (1 << PC5);
	PORTD |= (1 << PD0) | (1 << PD7);

	/* 8 bit 500Hz PWM */	
        TCCR1A |= (1<<WGM10);
	TCCR1A |= (1<<COM1A1) | (1<<COM1B1);
        TCCR1B |= (1 << CS10);
        /* 8 bit 500Hz */
        TCCR2 |= (1 << WGM20) | (1 << COM21);
	TCCR2 |= (1 << CS22) | (1 << CS21);
	
	encode_init();
	sei();
}

/* PB5 + OCRIGHT */
void move_right(uint8_t speed)
{
	OCLEFT = 0; 
	PORTB &= ~(1 << PB4);
	OCRIGHT = speed;
	PORTB |= (1 << PB5);	
}

/*PB4 + OCLEFT */
void move_left(uint8_t speed)
{
	OCRIGHT = 0; 
	PORTB &= ~(1 << PB5);
	OCLEFT = speed;
	PORTB |= (1 << PB4);	
}

void m_stop( void )
{
	PORTB &= ~(1 << PB4);
	PORTB &= ~(1 << PB5);
	OCRIGHT = 255; 
	OCLEFT = 255; 
}

/* basic PID controller */
void set_pid( void )
{
//	static int32_t int_sum = 0;
	static int32_t deltalast = 0;
	int64_t m_output = 0;

	int32_t delta = position_set - position ;
//	int_sum += delta;
/*
	if (int_sum > 20000) 
		int_sum = 20000;
	if (int_sum < -20000)
		int_sum = -20000;
*/
	m_output = PID_KP*delta + /* PID_KI*((int32_t)int_sum) +*/ (PID_KD*(delta-deltalast));
	deltalast = delta;


	if (m_output > 0) {
		if (m_output > 255) 
			move_right(255);
		else 
			move_right(m_output);
	} else if (m_output < 0){
		if (m_output < -255)
			move_left(255);
		else 
			move_left(-1*m_output);
	} else {
		m_stop();
	}

}

/*
 *  ISR must be fast enough to catch encoder at full speed
 */

ISR( TIMER0_OVF_vect )
{
	static uint16_t prescale = 0;
	cli();
	TCNT0 = 140; 
	int8_t new, diff;
	new = 0;
	if( PHASE_A )
	    new = 3;
	if( PHASE_B )
	    new ^= 1;
	diff = last - new;
	if ( diff & 1 ){
		last = new;
		position += (diff & 2) -1;
	}
	if (prescale < 1000){
		prescale++;
	} else {
		prescale = 0;
		PORTB |= (1 << PB0);
	}
	if (prescale == lid_pos)
		PORTB &= ~(1 << PB0);

	sei();
}

/* busy delay loop with PID set */
void busydelay(uint8_t cycle)
{
	for(;cycle > 0; cycle--){
		set_pid();
		_delay_ms(2);
	}
}

int main(void)
{
	_delay_ms(50);
	/* lcd port, TODO: move to the lcd_init() */	
	DDRD |= (1 << L_DB4) | (1 << L_DB5) | (1 << L_DB6) | (1 << L_DB7) | (1 << L_RS) | (1 << L_E);
	lcd_init();
	set_cursor(2,1);
	lcd_puts("   init... ");
	init();
	SERVO_IDLE;

	char number[8];

	/* move slider to max. left to set zero position */
	move_left(100);
	int16_t last_pos;
	do {
		last_pos = position;
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
		_delay_ms(100);
	} while (position != last_pos);
	m_stop();

	position = 0;

	position_set = 2000;

	while ( 1 ){
		uint8_t keyvar = 0;
		set_cursor(2,1);
		lcd_puts("               ");
		set_cursor(2,1);
		itoa(position, number, 10);
		lcd_puts(number);
		_delay_ms(2);


		/* lsb is SW0 */
		keyvar = ((PINC & 0x3F) << 2) | ((PIND & 1) << 1) | ((PIND & 128) >> 7);
		if (keyvar & 1) {
			busydelay(WAIT0);
			position_set = POS0;
			if( position >= (POS0 - POSTOL) && position <= (POS0 + POSTOL)){
				LIDOPEN;
				SERVO_TOG;
			} else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 2){
			busydelay(WAIT0);
			position_set = POS1;
			if( position >= (POS1 - POSTOL) && position <= (POS1 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 4){
			busydelay(WAIT0);
			position_set = POS2;
			if( position >= (POS2 - POSTOL) && position <= (POS2 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 8){
			busydelay(WAIT0);
			position_set = POS3;
			if( position >= (POS3 - POSTOL) && position <= (POS3 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 16){
			busydelay(WAIT0);
			position_set = POS4;
			if( position >= (POS4 - POSTOL) && position <= (POS4 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 32){
			busydelay(WAIT0);
			position_set = POS5;
			if( position >= (POS5 - POSTOL) && position <= (POS5 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 64){
			busydelay(WAIT0);
			position_set = POS6;
			if( position >= (POS6 - POSTOL) && position <= (POS6 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else if (keyvar & 128){
			busydelay(WAIT0);
			position_set = POS7;
			if( position >= (POS7 - POSTOL) && position <= (POS7 + POSTOL))
				SERVO_TOG;
			else if (keyvar){
				SERVO_HOLD;
				busydelay(WAIT1);
			}
		} else {
			SERVO_IDLE;
			LIDCLOSE;
			busydelay(25);
		}
	}

}

