/*
Heating Controller for a simple DIY reflow soldering oven with selectable
profiles. Developed on the work of Thomas Pfeifer (thomaspfeifer.net)
Copyright (C) 2015  Andreas Fiessler, fornax@leyanda.de

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

//#define oldgcc

#include <avr/io.h>
#include <stdio.h>
#include <stdlib.h>
#include <avr/interrupt.h>

#include "lcdlib.h"

#define BAUD_RATE 38400
#define DEBOUNCE 30

#ifdef oldgcc
  #include <avr/delay.h>
#else
  #include <util/delay.h>
#endif

#define clr_ledr()	PORTB |= (1 << PB0);
#define clr_ledg()	PORTB |= (1 << PB1);
#define clr_ledb()	PORTB |= (1 << PB2);
#define set_ledr()	PORTB &= ~(1 << PB0);
#define set_ledg()	PORTB &= ~(1 << PB1);
#define set_ledb()	PORTB &= ~(1 << PB2);

void adcinit(void);
int getadc(uint8_t channel);
void waitms(int ms);
void error(void);
void control(int t_ist, int t_soll);
int updateProfile(int t);

#define NUMPROFILES 3

int profile_t[10];
int profile_T[10];


volatile uint8_t rot_p_cnt;
volatile int8_t enc_delta;
volatile uint8_t is_pressed;
static int8_t last;

void encode_init( void )
{
        int8_t new;
        new = 0;
        if( (PINB & 1<<PB4) )
            new = 3;
        if( (PINB & 1<<PB3) )
            new ^= 1;
        last = new;
        enc_delta = 0;
        TCCR0 = (1 << CS01) | (1 << CS00);
        TIMSK |= 1<<TOIE0;
}

ISR( TIMER0_OVF_vect )
{
        TCNT0 = (uint8_t)(int16_t)-(F_CPU/150 * 10e-3 + 0.5); //1ms
        int8_t new, diff;
        new = 0;
        if( PINB & 1<<PB4 )
            new = 3;
        if( PINB & 1<<PB3 )
            new ^= 1;
        diff = last - new;
        if ( diff & 1 ){
                last = new;
                enc_delta += (diff & 2) -1;
        }

        if ( !( PINB & (1 << PB5)) ) {
                if ( rot_p_cnt < DEBOUNCE ){
                        rot_p_cnt++;
                } else {
                        is_pressed = 1;
                }
        } else {
                rot_p_cnt = 0;
                is_pressed = 0;
        }
}

int8_t encode_read(void)
{
        int8_t val;

        cli();
        val = enc_delta;
      	enc_delta = val & 3;
        sei();
        return val >> 2;
}


static int uart_putchar(char c)
{
	if (c == '\n')
		uart_putchar('\r');
	//Warten solange bis Zeichen gesendet wurde
	loop_until_bit_is_set(UCSRA, UDRE);
	//Ausgabe des Zeichens
	UDR = c;
	return 0;
}

void UART_Init (void)
{
	//Enable TXEN im Register UCR TX-Data Enable
	UCSRB=(1 << TXEN);
	//Teiler wird gesetzt
	UBRRL=(F_CPU / (BAUD_RATE * 16L) - 1);

#ifdef oldgcc
	//öffnet einen Kanal für printf (STDOUT)
	fdevopen (uart_putchar, NULL, 0);
	static int uart_putchar(char c, FILE *stream);
#else
	static FILE mystdout = FDEV_SETUP_STREAM(uart_putchar, NULL,_FDEV_SETUP_WRITE);
	stdout = &mystdout;
#endif
}

int convert2degree(int value)
{
	static const float c1=25;
	static const float v1=279;  

	static const float c2=213;
	static const float v2=149;  

	float m=((c1-c2)/(v1-v2));
	return((int)((m*(float)value)+(-m*v1)+c1));
}


int main (void)
{
	int adc;
	int tick=0;
	int t=0;
	float t_soll;

	adcinit();
	UART_Init();
	printf("ok\n");

	DDRC  = 0b00000000;

	DDRB  = 0b11000111;
	PORTB = 0b00111111;

	DDRD  = 0b11111100;  
	PORTD = 0b00000000; 
	encode_init();
	lcd_init();
	

	while ( 1 ){
		int8_t profile_select = 0;
		lcd_clr();
		set_cursor(0,2);
		lcd_puts(" T= ");

		clr_ledr();
		clr_ledb();
		set_ledg();

		while (((PINB & (1 << PB5))) ) {
			char line0[3];
			adc=getadc(0);
			dtostrf(convert2degree(adc), 3, 0, line0);
			//lcd_clr();
			int8_t encoder_val = encode_read();
			
			profile_select += encoder_val;
		
			/* NOTE: this ignores larger encoder values than 1 */	
			if(profile_select < 0){
				profile_select = NUMPROFILES - 1;
			} else if (profile_select > NUMPROFILES - 1){
				profile_select = 0;
			}
			


			set_cursor(0,1);
			switch(profile_select){
				case 0:
					lcd_puts("Set PbSn");
					break;
				case 1:
					lcd_puts("Set NoPb");
					break;
				case 2:
					lcd_puts("Set A520");
					break;
				default:
					lcd_puts("Prof N/A");
					break;
			}
			set_cursor(5,2);
			lcd_puts(line0);


		}
		switch(profile_select){
			case 0:

				profile_t[0]=0;
				profile_T[0]=15;

				profile_t[1]=100;
				profile_T[1]=150;

				profile_t[2]=210;
				profile_T[2]=183;

				profile_t[3]=220;
				profile_T[3]=210;

				profile_t[4]=250;
				profile_T[4]=210;

				profile_t[5]=260;
				profile_T[5]=0;

				profile_t[6]=400;
				profile_T[6]=0;
				break;
			case 1:
				profile_t[0]=0;
				profile_T[0]=15;

				profile_t[1]=100;
				profile_T[1]=150;

				profile_t[2]=210;
				profile_T[2]=183;

				profile_t[3]=250;
				profile_T[3]=245;

				profile_t[4]=280;
				profile_T[4]=245;

				profile_t[5]=280;
				profile_T[5]=0;

				profile_t[6]=400;
				profile_T[6]=0;
				break;
			case 2:
				profile_t[0]=0;
				profile_T[0]=15;

				profile_t[1]=100;
				profile_T[1]=120;

				profile_t[2]=160;
				profile_T[2]=130;

				profile_t[3]=240;
				profile_T[3]=160;

				profile_t[4]=242;
				profile_T[4]=160;

				profile_t[5]=280;
				profile_T[5]=138;

				profile_t[6]=400;
				profile_T[6]=0;
				break;
			default:
				error();
				break;
		}


		clr_ledg();
	
		set_cursor(0,2);
		lcd_puts("T");

		t_soll=0;
		t=0;
		while (t<400) {
			char line0[3];
			char line1[3];
			tick++;

			// ADC auslesen
			adc=getadc(0);
			if (adc<100 || adc>1000) error(); // sensor break/short detect

			// Regelung
			control(convert2degree(adc),t_soll);

			// 1/s t_soll update
			if (tick==50) {
				tick=0;
				t_soll=updateProfile(t);
				if (t_soll<-5 || t_soll>300) error();

				printf("%d,%d,%d\n",adc,(int)t_soll,convert2degree(adc));

				dtostrf(convert2degree(adc), 3, 0, line1);
				//dtostrf(adc, 3, 0, line1);
				dtostrf((int)t_soll, 3, 0, line0);
				//lcd_clr();
				set_cursor(1,2);
				lcd_puts(line0);
				lcd_puts("|");
				set_cursor(5,2);
				lcd_puts(line1);
				t++;
			}
			waitms(20);
		}
		PORTB &= ~((1 << PB6) | (1 << PB7));
	}
}

void error(void)
{
	lcd_clr();
	set_cursor(0,1);
	lcd_puts("error");
	while(1) {
		clr_ledr();
		clr_ledg();
		clr_ledb();
	    	waitms(600);
		set_ledr();
		waitms(600);
		printf("Error\n");
	}
}

void control(int t_ist, int t_soll)
{
	if (t_ist>t_soll) {
		PORTB &= ~((1<<PB6)|(1 << PB7));
		clr_ledr();
	} else {
		PORTB |= (1<<PB6)|(1 << PB7);
		set_ledr();
	}
}


float profil_t;
int profil;
float dt;

int updateProfile(int t)
{
	if (t==0) {
		profil=0;
	}
	switch(profil){
		case 1:
			set_cursor(0,1);
			lcd_puts("Preheat  ");
			break;
		case 2:
			set_cursor(0,1);
			lcd_puts("Flux act");
			break;
		case 3:
			set_cursor(0,1);
			lcd_puts("Heat max");
			break;
		case 4:
			set_cursor(0,1);
			lcd_puts("Solder  ");
			break;
		case 5:
		case 6:
			clr_ledr();
			set_ledb();
			set_cursor(0,1);
			lcd_puts("Cooldown");
			break;
		default: 
			break;
	}
	if (t==profile_t[profil]) {
		profil_t=profile_T[profil];

		dt=((float)profile_T[profil+1]-(float)profile_T[profil])/((float)profile_t[profil+1]-(float)profile_t[profil]);
		//printf("profil: %d\n",profil);
		profil++;
	} else {
		profil_t+=dt;
	}

	return((int)profil_t);
}

void adcinit(void)
{
  // Activate ADC with Prescaler 16 --> 1Mhz/16 = 62.5kHz
  //ADCSRA = _BV(ADEN) | _BV(ADPS2);
	ADCSRA = _BV(ADEN) | _BV(ADPS2) | _BV(ADPS1);
	ADMUX=  0;
}

int getadc(uint8_t channel)
{
	// Select pin ADC0 using MUX
	ADMUX = channel | _BV(REFS0) | _BV(REFS1);

	//Start conversion
	ADCSRA |= _BV(ADSC);

	// wait until converstion completed
	while (ADCSRA & _BV(ADSC) ) {}

	// get converted value
	return(ADCW);
}

void waitms(int ms)
{
	int i;
	for (i=0;i<ms;i++)
		_delay_ms(1);
}


